import streamlit as st
import pandas as pd
import datetime
import os
import numpy as np 

# --- Nombres de archivos para persistencia ---
DATA_FILE = "proyectos_y_tareas.csv"
DAILY_TASKS_FILE = "tareas_diarias.csv"

# --- Constantes para Estados y Categorías ---
ESTADOS_OPCIONES = {
    "Pendiente": "⚪ Pendiente",
    "Comenzado": "🟡 Comenzado",
    "Atrasado": "🔴 Atrasado",
    "Terminado": "🟢 Terminado"
}
DEFAULT_CATEGORIES = ["Trabajo", "Personal", "Estudios", "Hogar", "Otros"]

# --- Configuración de Streamlit ---
st.set_page_config(layout="wide", page_title="Organizador de Proyectos")

def obtener_opciones_estado():
    """Retorna las opciones de estado para Selectbox."""
    return list(ESTADOS_OPCIONES.values())

# --- Funciones de persistencia y manejo de estado ---

def cargar_datos(filename, default_data):
    """Carga datos de un archivo CSV, asegurando tipos de datos correctos."""
    if os.path.exists(filename):
        try:
            df = pd.read_csv(filename)
            
            # Conversiones de Fecha/Datetime
            for col in ['Fecha de Creación', 'Fecha de Vencimiento']:
                if col in df.columns:
                    df[col] = pd.to_datetime(df[col], errors='coerce') 

            if 'Completado' in df.columns:
                 df['Completado'] = df['Completado'].astype(bool)
            
            # Asegurar que existan las nuevas columnas para CSVs antiguos
            if filename == DATA_FILE:
                if 'Tiempo Real (horas)' not in df.columns:
                    df['Tiempo Real (horas)'] = np.nan
                if 'Fecha de Vencimiento' not in df.columns:
                    df['Fecha de Vencimiento'] = pd.NaT
                if 'Categoría' not in df.columns:
                    df['Categoría'] = 'Otros'

            return df
        except Exception as e:
            st.warning(f"Error al cargar {filename}: {e}. Se usarán datos por defecto.")
            return default_data
    else:
        return default_data

def guardar_datos():
    """Guarda los DataFrames de la sesión en archivos CSV."""
    if 'tasks' in st.session_state:
        # Guardar proyectos: eliminar columna de Desviación (calculada)
        df_to_save = st.session_state.tasks.copy().drop(columns=['Desviación (horas)'], errors='ignore')
        df_to_save.to_csv(DATA_FILE, index=False)
    
    if 'daily_tasks' in st.session_state:
        # Guardar tareas diarias: eliminar columna 'Completado (Hoy)' para forzar el reset diario.
        df_daily_to_save = st.session_state.daily_tasks.copy().drop(columns=['Completado (Hoy)'], errors='ignore')
        df_daily_to_save.to_csv(DAILY_TASKS_FILE, index=False)


def marcar_atrasadas(df):
    """Marca tareas como 'Atrasado' si la fecha de vencimiento ha pasado y no están completadas."""
    df_copy = df.copy()
    today = pd.Timestamp(datetime.date.today())
    
    condition = (df_copy['Fecha de Vencimiento'].notna()) & \
                (df_copy['Fecha de Vencimiento'].dt.date < today.date()) & \
                (df_copy['Completado'] == False) & \
                (df_copy['Estado'] != ESTADOS_OPCIONES['Terminado'])

    df_copy.loc[condition, 'Estado'] = ESTADOS_OPCIONES['Atrasado']
    return df_copy


def inicializar_datos():
    """Inicializa la tabla de datos y las tareas repetitivas en el estado de la sesión."""
    
    # --- PROYECTOS/TAREAS ---
    default_tasks = pd.DataFrame(
        {
            "Proyecto/Tarea": ["Configurar el entorno de desarrollo", "Diseñar el esquema de la base de datos"],
            "Tiempo Estimado (horas)": [4.0, 2.5],
            "Tiempo Real (horas)": [np.nan, 3.0], 
            "Fecha de Creación": [datetime.date.today(), datetime.date.today()], 
            "Fecha de Vencimiento": [datetime.date.today() + datetime.timedelta(days=7), pd.NaT], 
            "Categoría": ["Trabajo", "Estudios"], 
            "Estado": [ESTADOS_OPCIONES['Pendiente'], ESTADOS_OPCIONES['Comenzado']],
            "Completado": [False, False],
        }
    )

    default_tasks['Fecha de Creación'] = pd.to_datetime(default_tasks['Fecha de Creación'])
    default_tasks['Fecha de Vencimiento'] = pd.to_datetime(default_tasks['Fecha de Vencimiento'])

    if 'tasks' not in st.session_state:
        st.session_state.tasks = cargar_datos(DATA_FILE, default_tasks)
        if 'Estado' not in st.session_state.tasks.columns:
             st.session_state.tasks['Estado'] = ESTADOS_OPCIONES['Pendiente']

    # --- TAREAS DIARIAS (CON LÓGICA DE RESET Y CORRECCIÓN DE WARNING) ---
    default_daily_tasks_names = ["Revisar correo electrónico", "Planificar las 3 tareas más importantes"]
    
    if 'daily_tasks' not in st.session_state:
        
        df_daily_base = pd.DataFrame({'Tarea Diaria': default_daily_tasks_names}).copy()
        
        if os.path.exists(DAILY_TASKS_FILE):
             try:
                 df_daily_loaded = pd.read_csv(DAILY_TASKS_FILE)
                 if 'Tarea Diaria' in df_daily_loaded.columns:
                     # CORRECCIÓN: Usamos .copy() explícitamente para evitar el SettingWithCopyWarning
                     df_daily_base = df_daily_loaded[['Tarea Diaria']].copy()
             except Exception:
                 pass 
        
        # LÓGICA DE RESET DIARIO: Siempre añadir la columna 'Completado (Hoy)' y forzarla a False al inicio.
        df_daily_base['Completado (Hoy)'] = False
        st.session_state.daily_tasks = df_daily_base


def agregar_tarea(nombre, tiempo_estimado, fecha_vencimiento, categoria):
    """Agrega una nueva tarea/proyecto a la tabla principal."""
    
    vencimiento_val = pd.NaT if fecha_vencimiento is None else fecha_vencimiento

    nueva_fila = pd.DataFrame({
        "Proyecto/Tarea": [nombre],
        "Tiempo Estimado (horas)": [tiempo_estimado],
        "Tiempo Real (horas)": [np.nan], 
        "Fecha de Creación": [datetime.date.today()],
        "Fecha de Vencimiento": [vencimiento_val],
        "Categoría": [categoria],
        "Estado": [ESTADOS_OPCIONES['Pendiente']],
        "Completado": [False]
    })
    
    nueva_fila['Fecha de Creación'] = pd.to_datetime(nueva_fila['Fecha de Creación'])
    nueva_fila['Fecha de Vencimiento'] = pd.to_datetime(nueva_fila['Fecha de Vencimiento'])
    
    st.session_state.tasks = pd.concat([st.session_state.tasks, nueva_fila], ignore_index=True)
    guardar_datos()


# --- Inicialización y Lógica Principal ---
inicializar_datos()

# 1. Aplicar la lógica de atraso al estado principal antes de editar
st.session_state.tasks = marcar_atrasadas(st.session_state.tasks)


# --- Interfaz de Usuario de Streamlit ---

st.title("📝 Organizador Personal de Proyectos y Tareas")
st.markdown("Ahora con **seguimiento de tiempo, fechas de vencimiento y filtros avanzados**.")

# ----------------------------------------------------
# BARRA LATERAL: AÑADIR NUEVO PROYECTO
# ----------------------------------------------------

st.sidebar.header("➕ Añadir Nuevo Proyecto")

with st.sidebar.form("nueva_tarea_form"):
    nombre = st.text_input("Nombre del Proyecto/Tarea", max_chars=100)
    tiempo = st.number_input("Tiempo Estimado (horas)", min_value=0.1, max_value=100.0, step=0.1, value=1.0)
    
    fecha_vencimiento = st.date_input(
        "Fecha de Vencimiento (Opcional)", 
        min_value=datetime.date.today(), 
        value=None
    )
    categoria = st.selectbox("Categoría", options=DEFAULT_CATEGORIES)
    
    submitted = st.form_submit_button("Añadir a la Lista")

    if submitted and nombre:
        agregar_tarea(nombre, tiempo, fecha_vencimiento, categoria)
        st.rerun() 


# ----------------------------------------------------
# TAREAS DIARIAS (CON RESET AUTOMÁTICO)
# ----------------------------------------------------

st.header("🗓️ Tareas Repetitivas Diarias")
st.session_state.daily_tasks = st.data_editor(
    st.session_state.daily_tasks,
    column_config={
        "Tarea Diaria": st.column_config.TextColumn("Tarea Diaria", width="large"),
        "Completado (Hoy)": st.column_config.CheckboxColumn("Completado (Hoy)"),
    },
    num_rows="dynamic",
    hide_index=True,
    key="daily_editor"
)
guardar_datos()

st.divider()

# ----------------------------------------------------
# RESUMEN DE PRODUCTIVIDAD
# ----------------------------------------------------

st.header("📊 Resumen de Productividad")

total_estimado = st.session_state.tasks['Tiempo Estimado (horas)'].sum()
total_real = st.session_state.tasks['Tiempo Real (horas)'].sum()
total_pendiente = st.session_state.tasks.loc[st.session_state.tasks['Completado'] == False, 'Tiempo Estimado (horas)'].sum()

estado_counts = st.session_state.tasks['Estado'].value_counts().reindex(obtener_opciones_estado(), fill_value=0)

col1, col2, col3, col4 = st.columns(4)
col1.metric("Tareas Activas", estado_counts.drop(ESTADOS_OPCIONES['Terminado'], errors='ignore').sum())
col2.metric("Tiempo Estimado Total", f"{total_estimado:.1f} hr")
col3.metric("Tiempo Real Registrado", f"{total_real:.1f} hr")
col4.metric("Tiempo Pendiente (Est.)", f"{total_pendiente:.1f} hr")

st.subheader("Conteo por Estado")

# CORRECCIÓN DE DEPRECIACIÓN: use_container_width=True -> width='stretch'
st.dataframe(estado_counts.to_frame(name="Cantidad"), width='stretch')

st.divider()

# ----------------------------------------------------
# EDITOR PRINCIPAL DE PROYECTOS Y TAREAS (OPTIMIZADO)
# ----------------------------------------------------

st.header("🎯 Editor Principal de Proyectos y Tareas")

# OPTIMIZACIÓN: NO INCLUIR COLUMNAS CALCULADAS en el editor para evitar congelamiento.
df_edit = st.session_state.tasks.copy() 

# CONFIGURACIÓN DE ALTURA PARA FORZAR PAGINACIÓN Y ESTABILIDAD
ALTURA_ESTABLE = 600

edited_df = st.data_editor(
    df_edit, 
    column_config={
        "Proyecto/Tarea": st.column_config.TextColumn("Proyecto/Tarea", width="large"),
        "Tiempo Estimado (horas)": st.column_config.NumberColumn("Tiempo Estimado (horas)", format="%.1f hr", min_value=0.1),
        "Tiempo Real (horas)": st.column_config.NumberColumn("Tiempo Real (horas)", format="%.1f hr", min_value=0.0, help="Tiempo real dedicado a la tarea."),
        "Fecha de Creación": st.column_config.DateColumn("Fecha de Creación", format="DD/MM/YYYY", disabled=True), 
        "Fecha de Vencimiento": st.column_config.DateColumn("Fecha de Vencimiento", format="DD/MM/YYYY", help="Fecha límite para completar la tarea."),
        "Categoría": st.column_config.SelectboxColumn("Categoría", options=DEFAULT_CATEGORIES, default="Otros", help="Área a la que pertenece la tarea."),
        "Estado": st.column_config.SelectboxColumn(
            "Estado",
            options=obtener_opciones_estado(),
            default=ESTADOS_OPCIONES['Pendiente'],
            required=True
        ),
        "Completado": st.column_config.CheckboxColumn("Completado"),
        # NOTA: La columna "Desviación (horas)" SE ELIMINÓ DEL EDITOR para optimización.
    },
    num_rows="dynamic", 
    hide_index=True,
    key="tasks_editor",
    height=ALTURA_ESTABLE # *** LÍNEA CLAVE PARA ESTABILIDAD ***
)

# 2. Reemplazar el estado de la sesión con los datos editados
st.session_state.tasks = edited_df
guardar_datos()

# ----------------------------------------------------
# VISTA FILTRADA (NO EDITABLE, MUESTRA DESVIACIÓN)
# ----------------------------------------------------

st.subheader("Filtro de Vista")

col_filter_1, col_filter_2, col_filter_3 = st.columns(3)

with col_filter_1:
    show_completed = st.checkbox("Mostrar tareas completadas en la vista", value=False)
with col_filter_2:
    selected_status = st.selectbox("Filtrar por Estado", options=["Todos"] + obtener_opciones_estado(), index=0, key="filter_status")
with col_filter_3:
    selected_category = st.selectbox("Filtrar por Categoría", options=["Todas"] + DEFAULT_CATEGORIES, index=0, key="filter_category")

# 1. CALCULAR DESVIACIÓN ANTES DE FILTRAR
df_display_filtered = edited_df.copy()
df_display_filtered['Desviación (horas)'] = df_display_filtered['Tiempo Real (horas)'] - df_display_filtered['Tiempo Estimado (horas)']
df_display_filtered['Desviación (horas)'] = df_display_filtered['Desviación (horas)'].round(1)


# 2. APLICAR FILTROS
if not show_completed:
    df_display_filtered = df_display_filtered[df_display_filtered['Completado'] == False]
if selected_status != "Todos":
    df_display_filtered = df_display_filtered[df_display_filtered['Estado'] == selected_status]
if selected_category != "Todas":
    df_display_filtered = df_display_filtered[df_display_filtered['Categoría'] == selected_category]


st.caption("A continuación se muestra la vista filtrada (usa el editor principal para modificar):")

# CORRECCIÓN DE DEPRECIACIÓN: use_container_width=True -> width='stretch'
st.dataframe(df_display_filtered, hide_index=True, width='stretch')